<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * List products. One widget to rule them all.
 *
 * @package  Nailhammer
 * @category Widgets
 * @extends  WC_Widget
 */
if ( ! class_exists( 'Nailhammer_Widget_Products' ) ) {

	class Nailhammer_Widget_Products extends WC_Widget {

		/**
		 * Constructor.
		 */
		public function __construct() {
			$this->widget_cssclass    = 'woocommerce' . ' ' . 'nailhammer_widget_products';
			$this->widget_description = esc_html__( 'Display a list of your products on your site.', 'nailhammer' );
			$this->widget_id          = 'nailhammer_products';
			$this->widget_name        = esc_html__( 'Nailhammer Products', 'nailhammer' );
			$this->settings           = array(
				'title'           => array(
					'type'  => 'text',
					'std'   => esc_html__( 'Products', 'nailhammer' ),
					'label' => esc_html__( 'Title', 'nailhammer' )
				),
				'number'          => array(
					'type'  => 'number',
					'step'  => 1,
					'min'   => 1,
					'max'   => '',
					'std'   => 5,
					'label' => esc_html__( 'Number of products to show', 'nailhammer' )
				),
				'show'            => array(
					'type'    => 'select',
					'std'     => '',
					'label'   => esc_html__( 'Show', 'nailhammer' ),
					'options' => array(
						'featured' => esc_html__( 'Featured Products', 'nailhammer' ),
						'onsale'   => esc_html__( 'On-sale Products', 'nailhammer' ),
					)
				),
				'orderby'         => array(
					'type'    => 'select',
					'std'     => 'date',
					'label'   => esc_html__( 'Order by', 'nailhammer' ),
					'options' => array(
						'date'  => esc_html__( 'Date', 'nailhammer' ),
						'price' => esc_html__( 'Price', 'nailhammer' ),
						'rand'  => esc_html__( 'Random', 'nailhammer' ),
						'sales' => esc_html__( 'Sales', 'nailhammer' ),
					)
				),
				'order'           => array(
					'type'    => 'select',
					'std'     => 'desc',
					'label'   => _x( 'Order', 'Sorting order', 'nailhammer' ),
					'options' => array(
						'asc'  => esc_html__( 'ASC', 'nailhammer' ),
						'desc' => esc_html__( 'DESC', 'nailhammer' ),
					)
				),
				'hide_free'       => array(
					'type'  => 'checkbox',
					'std'   => 0,
					'label' => esc_html__( 'Hide free products', 'nailhammer' )
				),
				'show_hidden'     => array(
					'type'  => 'checkbox',
					'std'   => 0,
					'label' => esc_html__( 'Show hidden products', 'nailhammer' )
				),
				'show_categories' => array(
					'type'  => 'checkbox',
					'std'   => 0,
					'label' => esc_html__( 'Show categories', 'nailhammer' )
				),
				'show_rating' => array(
					'type'  => 'checkbox',
					'std'   => 0,
					'label' => esc_html__( 'Show rating', 'nailhammer' )
				),
				'show_quick_view_compare_wishlist' => array(
					'type'  => 'checkbox',
					'std'   => 0,
					'label' => esc_html__( 'Show quick and wishlist & compare block', 'nailhammer' )
				),
				'show_add_to_cart_button' => array(
					'type'  => 'checkbox',
					'std'   => 0,
					'label' => esc_html__( 'Show add to cart button', 'nailhammer' )
				)
			);

			parent::__construct();
		}

		/**
		 * Query the products and return them.
		 *
		 * @param  array $args
		 * @param  array $instance
		 *
		 * @return WP_Query
		 */
		public function get_products( $args, $instance ) {
			$number  = ! empty( $instance['number'] ) ? absint( $instance['number'] ) : $this->settings['number']['std'];
			$show    = ! empty( $instance['show'] ) ? sanitize_title( $instance['show'] ) : $this->settings['show']['std'];
			$orderby = ! empty( $instance['orderby'] ) ? sanitize_title( $instance['orderby'] ) : $this->settings['orderby']['std'];
			$order   = ! empty( $instance['order'] ) ? sanitize_title( $instance['order'] ) : $this->settings['order']['std'];

			$query_args = array(
				'posts_per_page' => $number,
				'post_status'    => 'publish',
				'post_type'      => 'product',
				'no_found_rows'  => 1,
				'order'          => $order,
				'meta_query'     => array()
			);

			if ( empty( $instance['show_hidden'] ) ) {
				array_push($query_args['meta_query'], WC()->query->visibility_meta_query() );
				$query_args['post_parent']  = 0;
			}

			if ( ! empty( $instance['hide_free'] ) ) {
				array_push( $query_args['meta_query'], array(
					'key'     => '_price',
					'value'   => 0,
					'compare' => '>',
					'type'    => 'DECIMAL',
				) );
			}

			array_push( $query_args['meta_query'], WC()->query->stock_status_meta_query() );
			$query_args['meta_query']   = array_filter( $query_args['meta_query'] );

			switch ( $show ) {
				case 'featured' :
					array_push( $query_args['meta_query'], array(
						'key'   => '_featured',
						'value' => 'yes'
					) );
					break;
				case 'onsale' :
					$product_ids_on_sale    = wc_get_product_ids_on_sale();
					$product_ids_on_sale[]  = 0;
					$query_args['post__in'] = $product_ids_on_sale;
					break;
			}

			switch ( $orderby ) {
				case 'price' :
					$query_args['meta_key'] = '_price';
					$query_args['orderby']  = 'meta_value_num';
					break;
				case 'rand' :
					$query_args['orderby'] = 'rand';
					break;
				case 'sales' :
					$query_args['meta_key'] = 'total_sales';
					$query_args['orderby']  = 'meta_value_num';
					break;
				default :
					$query_args['orderby'] = 'date';
			}

			return new WP_Query( apply_filters( 'woocommerce_products_widget_query_args', $query_args ) );
		}

		/**
		 * Output widget.
		 *
		 * @see WP_Widget
		 *
		 * @param array $args
		 * @param array $instance
		 */
		public function widget( $args, $instance ) {
			$template = locate_template( 'inc/widgets/products/views/products-widget-views.php', false, false );

			if ( ( $products = $this->get_products( $args, $instance ) ) && $products->have_posts() && $template ) {
				$this->widget_start( $args, $instance );

				echo apply_filters( 'woocommerce_before_widget_product_list', '<ul class="product_list_widget">' );

				while ( $products->have_posts() ) {
					$products->the_post();
					include $template;
				}

				echo apply_filters( 'woocommerce_after_widget_product_list', '</ul>' );

				$this->widget_end( $args );
			}

			wp_reset_postdata();
		}
	}

	add_action( 'widgets_init', 'nailhammer_register_widget_products' );
	/**
	 * Register products widget.
	 */
	function nailhammer_register_widget_products() {
		register_widget( 'Nailhammer_Widget_Products' );
	}
}