<?php

/**
 * NailHammer Image Custom Menu
 *
 * @author   TemplateMonster
 * @category Widgets
 * @version  1.0.0
 * @extends  WP_Nav_Menu_Widget
 */

if ( class_exists( 'WP_Nav_Menu_Widget' ) ) {
	
	class NailHammer_Image_Custom_Menu_Widget extends WP_Nav_Menu_Widget {
		
		/**
		 * Sets up a new NailHammer Image Custom Menu widget instance.
		 *
		 * @since 1.0.0
		 */
		public function __construct() {
			
			$widget_ops = array(
				'description' => esc_html__( 'Add a custom menu with image.', 'nailhammer' )
			);
			
			add_action( 'admin_enqueue_scripts', array( $this, 'register_admin_assets' ) );
			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
			
			WP_Widget::__construct( 'image_custom_menu_widget', esc_html__( 'Image Custom Menu', 'nailhammer' ), $widget_ops );
		}
		
		public function register_admin_assets() {
			wp_register_style( 'image-custom-menu-widget-admin', NAILHAMMER_THEME_URI . '/inc/widgets/image-custom-menu/admin/assets/css/image-custom-menu-widget-admin.css', array( 'dashicons' ) );
			wp_register_script( 'image-custom-menu-widget-admin', NAILHAMMER_THEME_URI . '/inc/widgets/image-custom-menu/admin/assets/js/image-custom-menu-widget-admin.js', array( 'jquery' ), NAILHAMMER_THEME_VERSION, true );
			$translation_custom_menu = array(
				'mediaFrameTitle' => esc_html__( 'Choose background image', 'nailhammer' )
			);
			wp_localize_script( 'image-custom-menu-widget-admin', 'customMenuWidgetAdmin', $translation_custom_menu );
		}
		
		/**
		 * Enqueue admin assets.
		 *
		 * @since 1.0.0
		 * @return void
		 */
		public function enqueue_admin_assets() {
			wp_enqueue_style( 'image-custom-menu-widget-admin' );
			wp_enqueue_script( 'image-custom-menu-widget-admin' );
		}
		
		/**
		 * Outputs the content for the current Image Custom Menu widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $args
		 * @param array $instance
		 */
		public function widget( $args, $instance ) {
			
			$id            = ! empty( $instance['id'] ) ? $instance['id'] : '';
			$class         = isset( $instance['class'] ) ? $instance['class'] : '';
			$img = '';
			
			if ( '' !== $id ) {
				$img = wp_get_attachment_image_src( $id, 'original' );
			}
			
			
			if ( is_array( $img ) ) {
				$html = '<img src="' . $img[0] . '" width="' . $img[1] . '" height="' . $img[2] . '" alt=" ' . $img[3] . '" >';
			} else {
				$html = '';
			}
			
			$args['before_widget'] .= '<div class="image_custom_menu_widget ' . $class . '"> ' . $html;
			$args['after_widget']   = '</div>' . $args['after_widget'];
			
			parent::widget( $args, $instance ); ?>
		
		<?php }
		
		/**
		 * Handles updating settings for the current Image Custom Menu widget instance.
		 *
		 * @since 1.0.0
		 *
		 * @param array $new_instance
		 * @param array $old_instance
		 *
		 * @return array
		 */
		public function update( $new_instance, $old_instance ) {
			
			$instance                  = parent::update( $new_instance, $old_instance );
			$instance['id']            = ! empty( $new_instance['id'] ) ? ( int ) $new_instance['id'] : 0;
			
			if ( isset( $new_instance['class'] ) ) {
				$instance['class'] = sanitize_text_field( $new_instance['class'] );
			}

			return $instance;
		}
		
		/**
		 * Outputs the settings form for the Custom Menu widget.
		 *
		 * @since 1.0.0
		 *
		 * @param array $instance
		 */
		public function form( $instance ) {
			
			parent::form( $instance );
			
			$menus = wp_get_nav_menus();
			
			if ( ! empty( $menus ) ) {
				
				$id    = ! empty( $instance['id'] ) ? $instance['id'] : '';
				$class = isset( $instance['class'] ) ? $instance['class'] : '';
				$img   = '';
				
				if ( '' !== $id ) {
					
					$img = wp_get_attachment_image_src( $id, 'thumbnail' );
				}
				
				?>
				<div class="image-custom-menu-widget-form-controls">
					<div class="image_custom_menu_widget_img"<?php if ( '' === $id ) { ?> style="display: none;"<?php } ?>>
						<div<?php if ( '' !== $id && is_array( $img ) ) {
							echo ' style="background-image: url(' . $img[0] . ');"';
						} ?>>
						<span class="banner_remove">
							<span class="dashicons dashicons-dismiss"></span>
						</span>
						</div>
					</div>
					<div class="image_custom_menu_widget_add_media"<?php if ( '' !== $id ) { ?> style="display: none;"<?php } ?>>
					<span>
						<span><?php esc_attr_e( 'Choose background image', 'nailhammer' ); ?></span>
					</span>
					</div>
					<input autocomplete="off" class="image_custom_menu_widget_id" type="hidden" id="<?php echo esc_attr( $this->get_field_id( 'id' ) ); ?>" name="<?php echo $this->get_field_name( 'id' ); ?>" value="<?php echo $id; ?>">
				</div>
				<p>
					<label for="<?php echo $this->get_field_id( 'class' ); ?>"><?php esc_html_e( 'Custom class:', 'nailhammer' ) ?></label>
					<input type="text" class="widefat" id="<?php echo $this->get_field_id( 'class' ); ?>" name="<?php echo $this->get_field_name( 'class' ); ?>" value="<?php echo $class ; ?>"/>
				</p>
			<?php }
		}
		
		
	}
	
	add_action( 'widgets_init', 'nailhammer_register_widget_image_custom_menu' );
	
	/**
	 * Register custom menu with image widget.
	 */
	function nailhammer_register_widget_image_custom_menu() {
		register_widget( 'NailHammer_Image_Custom_Menu_Widget' );
	}
}
